/* Copyright (c) 2002, 2011, Oracle and/or its affiliates. 
All rights reserved. */


/* Copyright (c) 2002, 2011, Oracle and/or its affiliates. 
All rights reserved. */

/*
    NAME
        PhotoAlbumServlet.java

    DESCRIPTION
        PhotoAlbumServlet is a Java Servlet that demonstrates the use
        of Oracle Multimedia Java Classes for Servlets and JSPs by implementing a
        simple photo album application.

    NOTES
        See the README.txt file for information on how to build and
        run the demo.
*/

import java.util.Stack;
import java.util.Enumeration;
import java.io.IOException;
import java.io.PrintWriter;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import javax.servlet.ServletException;
import javax.servlet.ServletConfig;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import oracle.jdbc.OraclePreparedStatement;
import oracle.jdbc.OracleResultSet;
import oracle.ord.im.OrdImage;
import oracle.ord.im.OrdHttpUploadFile;
import oracle.ord.im.OrdHttpUploadFormData;
import oracle.ord.im.OrdHttpResponseHandler;

/**
 * PhotoAlbumServlet is a Java Servlet that demonstrates the use of
 * Oracle Multimedia Java Classes for Servlets and JSPs by implementing a
 * simple photo album application.
 */
public class PhotoAlbumServlet extends HttpServlet
{
    //static variables for database connection
    private static String s_username = null;
    private static String s_password = null;
    private static String s_connString = null;

    private static final String ERRORMSG = "Please provide a valid database "; 
    //
    // A Java Stack used to implement a simple connection pooling mechanism
    //
    private static Stack connStack = new Stack();

    //
    // Flag indicating if the JDBC driver has been loaded.
    //
    private static boolean driverLoaded = false;

    public final static String DB_CONN_FAIL = 
        "An attempt to connect to the database has failed. " +
        "Please try again.";

    /**
     * Servlet initialization method.
     */
    public void init( ServletConfig config ) throws ServletException
    {
        super.init(config);
    }

    /*
     * Process an HTTP Get request. The following requests are processed by
     * this method:
     * - media: retrieve a full-size or thumbnail image
     * - new: display a form allowing a new entry to be uploaded
     * - view: view full-size image
     * - album: view photo album and thumbnail images
     * - default: display a form requiring username, password and connString
     */
    public void doGet( HttpServletRequest request,
                       HttpServletResponse response )
        throws ServletException, IOException
    {
        Connection conn = null;
        PhotoAlbumRequest albumRequest = null;
        //
        // Use a try-block to ensure that JDBC connections are always returned
        // to the pool.
        //
        try
        {
            //
            //the first time initiating the servlet, no database connection
            //
            if(!((Enumeration)request.getParameterNames()).hasMoreElements())
                albumRequest = new PhotoAlbumRequest( null, request, response );
            else{
                //
                //Get a JDBC connection from the pool
                //
                conn = getConnection();
                //
                // Instantiate an PhotoAlbumRequest object to process the request.
                //
                albumRequest =
                    new PhotoAlbumRequest( conn, request, response );
            }        
            //
            // Figure out what to do based on query string parameters.
            //
            String view_media = request.getParameter( "view_media" );
            if ( view_media != null )
            {
                //
                // Deliver a full-size or thumbnail image to the browser.
                //
                albumRequest.viewMedia( view_media );
                return;
            }
            else if ( request.getParameter( "view_form" ) != null )
            {
                //
                // Display the HTML upload form.
                //
                albumRequest.viewUploadForm();
            }
            else if ( request.getParameter( "view_entry" ) != null )
            {
                //
                // Display full-size photo image.
                //
                albumRequest.viewPhoto();
            }
            else if( request.getParameter( "view_album" ) != null)
            {
                //
                // Display album contents with thumbnails by default.
                //
                albumRequest.viewAlbum();
            }
            else
            {
              //
              //Display the DB connection inforamtion page
              //
              //Empty the stack first
              //
              emptyStack();
              //
              //Read the parameters from previous page
              //
              String username = request.getParameter("ousername");
              String connString = request.getParameter("oconnString");
              String errorMessage = request.getParameter("error");
              albumRequest.viewDBInfoForm(username, connString, errorMessage);  
            }   
        }
        catch ( SQLException e )
        {
            //
            // Log what went wrong.
            //
            e.printStackTrace( System.out );

            //
            // Turn SQL exceptions into ServletExceptions.
            //
            throw new ServletException( e.toString() );
        }
        finally
        {
            //
            // If we have a JDBC connection, then return it to the pool.
            //
            freeConnection( conn );
        }
    }

    /*
     * Process an HTTP Post request used to 
     * -- set db username, password and connstr
     * -- upload a new photo into the album
     */
    public void doPost( HttpServletRequest request,
                        HttpServletResponse response )
        throws ServletException, IOException
    {
        Connection conn = null;
        //
        //Process the DB connection inforamtion form
        //
        if(request.getParameter( "page") != null)
        {
            PhotoAlbumRequest albumRequest =
                new PhotoAlbumRequest( null, request, response );

            String username = request.getParameter("username");
            String password = request.getParameter("password"); 
            String connString = request.getParameter("connString");
            //
            //error handling if any of the username, password or connection
            //string is missing
            //
            if(username == null || username.length()==0){
                albumRequest.viewDBInfoForm( username, connString, 
                    ERRORMSG+"username.");
            }
            else if(password == null || password.length() ==0){
                albumRequest.viewDBInfoForm( username, connString, 
                    ERRORMSG+"password.");
            }
            else if(connString == null || connString.length() ==0){
                albumRequest.viewDBInfoForm( username, connString, 
                    ERRORMSG+"connection string.");
            }
            //
            //Set up DB connection values and create a connection
            //
            else{
                setUsername(username.trim()); 
                setPassword(password.trim());
                setConnString(connString.trim());
                try{
                    conn = getConnection();
                    albumRequest.setConnection(conn);
                    //
                    //display photoAlbum
                    //
                    albumRequest.viewAlbum();
                }catch(SQLException e){
                    //
                    //handle connection fail exception
                    //
                    String dbConnErrMsg = e.getMessage();
                    if(dbConnErrMsg.compareTo(DB_CONN_FAIL) == 0){
                        albumRequest.viewDBInfoForm( username, connString, 
                            dbConnErrMsg);                        
                    }
                    else{
                        //
                        // Log what went wrong.
                        //
                        e.printStackTrace( System.out );
                        //
                        // Turn SQL exceptions into ServletExceptions.
                        //
                        throw new ServletException( e.toString() );
                    }
                }
                finally
                {
                    //
                    // If we have ba JDBC connection, then return it to the pool.
                    //
                    freeConnection( conn );
                }
            }
        }
        
        else if(request.getParameter( "upload") != null)
        {
        //
        // Use a try-block to ensure that JDBC connections are always returned
        // to the pool.
        //
            try
            {
                //
                // Get a JDBC connection from the pool
                //
                conn = getConnection();

                //
                // Instantiate an PhotoAlbumRequest object to process the request
                //
                PhotoAlbumRequest albumRequest =
                    new PhotoAlbumRequest( conn, request, response );

                //
                // Insert the photo into the album.
                //
                albumRequest.insertNewPhoto();
            }
            catch ( SQLException e )
            {
                //
                // Log what went wrong.
                //
                e.printStackTrace( System.out );

                //
                // Turn SQL exceptions into ServletExceptions.
                //
                throw new ServletException( e.toString() );
            }
            finally
            {
                //
                // If we have ba JDBC connection, then return it to the pool.
                //
                freeConnection( conn );
            }
        }
    }

    /*
     * Get Servlet information
     */
    public String getServletInfo()
    {
        return "Oracle Multimedia Java Servlet Photo Album Demo";
    }


    /**
     * Remove all the connections in the stack
     */
    public void emptyStack(){
        synchronized( connStack )
        {
            connStack.clear();
        }      
    }

    /** 
     * set the database connection username 
     */
    public void setUsername(String username){
        s_username = username;
    }

    /** 
     * set the database connection password 
     */
    public void setPassword(String password){
        s_password = password;
    }

    /** 
     * set the database connection string 
     */
    public void setConnString(String connString){
        s_connString = connString;
    }

    /**
     * The getConnection method implements a simple JDBC connection pool using
     * a Java Stack to hold the set of available connections. If the stack is
     * empty, then getConnection simply creates a new connection.
     */
    private Connection getConnection()
        throws SQLException
    {
        Connection conn = null;

        //
        // Synchonize on the Stack object. Load the JDBC driver if not yet
        // done. If there is a free connection on the stack, then pop it off
        // the stack and return it to the caller. Otherwise, create a new
        // connection object.
        //
        synchronized( connStack )
        {
            if ( !driverLoaded )
            {
                DriverManager.registerDriver(
                                    new oracle.jdbc.OracleDriver() );
                driverLoaded = true;
            }
            if ( connStack.empty() )
            {
                try{
                    conn = DriverManager.getConnection
                    ( s_connString, s_username, s_password );
                }catch(SQLException sqle){
                    throw new SQLException(DB_CONN_FAIL);
                }
            }
            else
            {
                conn = (Connection)connStack.pop();
            }
        }

        //
        // Enable auto-commit by default.
        //
        conn.setAutoCommit( true );

        return conn;
    }

    /**
     * The freeConnection method simply returns a JDBC connection to the pool.
     */
    private void freeConnection( Connection conn )
    {
        //
        // Synchonize on the Stack object, then push the connection onto the
        // stack.
        //
        if ( conn != null )
        {
            synchronized( connStack )
            {
                connStack.push( conn );
            }
        }
    }
}

/**
 * The PhotoAlbumRequest class is responsible for the actual processing of
 * an HTTP GET or POST request.
 */
class PhotoAlbumRequest
{
    //
    // Request execution information.
    //
    private Connection conn;
    private HttpServletRequest request;
    private HttpServletResponse response;
    private PrintWriter out;
    private String servletUri;

    //
    // The 'constructor' for an empty ORDSYS.ORDImage object.
    //
    private final static String EMPTY_IMAGE = "ordsys.ordimage.init()";

    /**
     * Constructor
     */
    PhotoAlbumRequest( Connection conn,
                       HttpServletRequest request,
                       HttpServletResponse response )
    {
        this.conn = conn;
        this.request = request;
        this.response = response;
        this.out = null;

        //
        // Get the servlet's URI. This shouldn't be necessary, since a servlet
        // should be able to use a self-relative URI to reference itself;
        // however, this doesn't work in JDeveloper, where the URL is always
        // /servlets/servlet.
        //
        servletUri = request.getRequestURI();
    }

    /** 
     * set the database connection
     */
    void setConnection(Connection conn){
        this.conn = conn;
    }


    /**
     * Display the contents of the photo album with thumbnail images for all
     * the entries.
     */
    void viewAlbum()
        throws ServletException, IOException, SQLException
    {
        int rowCount = 0;

        //
        // Write a common HTML header.
        //
        printPageHeader();

        //
        // Prepare and execute a SQL statement to fetch the ID, description,
        // location and thumbnail image for all the entries in the album.
        // The thumbnail image for each entry is used to build the height
        // and width attributes for the image tag.
        //
        PreparedStatement stmt =
            conn.prepareStatement( "select id,description,location,thumb " +
                                   " from photos order by description" );
        OracleResultSet rset = (OracleResultSet)stmt.executeQuery();

        //
        // Display the thumbnail images in a table.
        //
        out.println( "<p><table cellpadding=\"3\" cellspacing=\"0\"" +
                     " border=\"1\" width=\"100%\"" + 
                     " summary=\"Table of thumb nail images\">" );
        out.println( "<tr bgcolor=\"#336699\">" );
        out.println( "<th id=\"description\"><font color=\"#ffffff\">" + 
                     "Description</font></th>" );
        out.println( "<th id=\"location\"><font color=\"#ffffff\">" +
                     "Location</font></th>" );
        out.println( "<th id=\"image\"><font color=\"#ffffff\">Image" +
                     "</font></th></tr>" );

        //
         // For each entry...
        //
        while ( rset.next() )
        {
            //
            // Increment the row count so we know if there are any entries
            // in the album.
            //
            rowCount++;

            //
            // Get the entry ID for building URLs.
            //
            String id = rset.getString( 1 );

            //
            // Start the table row and display the photo description and
            // location.
            //
            String description = rset.getString( 2 );
            description = escapeHtmlString(description);

            out.println( "<tr>" );
            out.println( "<td headers=\"description\">" + description +
                         "</td>" );
            String location = rset.getString( 3 );
            if ( location == null )
            {
                location = "&nbsp;";
            }else
            {
                location = escapeHtmlString(location);
            }
            out.println( "<td headers=\"location\">" + location + "</td>" );

            //
            // Get the thumbnail OrdImage object from the result set so we
            // can build the height and width attributes in the image tag.
            //
            OrdImage img = (OrdImage)rset.getORAData( 
                                         4, OrdImage.getORADataFactory() );

            //
            // Display the thumbnail image as an anchor tag which can be used
            // to display the full-size image. If the image format isn't
            // supported by Oracle Multimedia, then a thumbnail wouldn't have been
            // produced when the image was uploaded, so use the text "[view
            // image]" instead of the thumbnail.
            //
            out.print( "<td headers=\"image\"><a href=\"" + servletUri +
                       "?view_entry=yes&id=" + id + "\">" );
            if ( img.getContentLength() > 0 )
            {
                if (img.getMimeType().startsWith("image/"))
                {
                out.print( "<img src=\"" + servletUri +
                           "?view_media=thumb&id=" + id + "\"" +
                           " height=" + img.getHeight() +
                           " width=" + img.getWidth() +
                           " alt=\"" + description + "\"" +
                           " border=1>" );
                }
            }
            else
            {
                out.print( "[view media]" );
            }
            out.println( "</a></td>" );
            out.println( "</tr>" );
        }

        //
        // Display a message if the album is empty
        //
        out.println( "<tr><td scope=\"col\" colspan=\"3\" align=\"center\">" +
                     "<font color=\"#336699\">" );
        if ( rowCount == 0 )
        {
            out.println( "<b><i>The photo album is empty</i></b>" );
        }
        else
        {
            out.println( "<b><i>Select the thumbnail to view the" +
                         " full-size image</i></b>" );
        }
        out.println( "</font></td></tr>" );
        out.println( "</table>" );

        //
        // Display an anchor tag that will display the new entry form.
        //
        out.println("<p> <table width=\"100%\" align=\"center\">");
        out.println("<tr bgcolor=\"#f7f7e7\"><td align=\"center\">");
        out.println("<a href=\""+servletUri+"\">Change schema</a></td>");
        out.println("<td align=\"center\">");
        out.println("<a href=\""+servletUri+"?view_form=yes\">Upload new photo</a>");
        out.println("</td></tr></table></p>");

        //
        // Write the common HTML trailer
        //
        printPageTrailer( false );

        //
        // Close the result-set and the statement.
        //
        rset.close();
        stmt.close();
    }

    /**
     * View a full-size photo
     */
    void viewPhoto()
        throws ServletException, IOException, SQLException
    {
        //
        // Write a common HTML header.
        //
        printPageHeader();

        //
        // Get the ID of the entry being viewed.
        //
        String id = request.getParameter( "id" );

        //
        // Prepare and execute a SQL statement to fetch the entry's
        // description, location and full-size image. The full-size
        // image is used to build the height and width attributes of
        // the image tag. Make sure the entry exists.
        //
        PreparedStatement stmt =
            conn.prepareStatement( "select description, location, image " +
                                   " from photos where id = ?" );
        stmt.setString( 1, id );
        OracleResultSet rset = (OracleResultSet)stmt.executeQuery();
        if ( !rset.next() )
        {
            throw new ServletException( "row not found in table" );
        }

        //
        // Get the thumbnail OrdImage object from the result set so we
        // can build the height and width attributes in the image tag.
        //
        OrdImage img =
            (OrdImage)rset.getORAData( 3, OrdImage.getORADataFactory() );


        //
        // Display the data in a table, starting with the description and
        // location.
        //
        out.println( "<p><table summary=\"View image entry\">" );
        String description = rset.getString( 1 );
        description = escapeHtmlString(description);
        
        out.println( "<tr><td scope=\"col\" ><b>Description:</b></td>" +
                     "<td scope=\"col\" >" +
                     description + "</td></tr>" );
        String location = rset.getString( 2 );
        if ( location == null )
        {
            location = "&nbsp;";
        }else
        {
            location = escapeHtmlString(location);
        }
        out.println( "<tr><td scope=\"col\" ><b>Location:</b></td>" +
                     "<td scope=\"col\" >" +
                     location + "</td></tr>" );

        //
        // Build a URL to fetch the full-size image for this entry. If the
        // image format is not recognized, then don't generate the height
        // and width attributes for the image tag. Specifying the height
        // and width attributes allows the browser to render the page without
        // having to wait to retrieve the image to determine its size, but
        // are not required.
        //
        out.println( "<tr><td scope=\"col\" valign=\"top\"><b>Photo:</b></td>" +
                     "<td scope=\"col\" >" );
        if (img.getMimeType().startsWith("image/"))
        {
            out.print( "<img src=\"" + servletUri +
                       "?view_media=image&id=" + id + "\"" );
            if ( img.getHeight() > 0 && img.getWidth() > 0 )
            {
                out.print( " height=" + img.getHeight() +
                       " width=" + img.getWidth() );
            }
            out.println( " alt=\"" + description + "\"" );
            
            out.println( " border=1>");
        }
        else
        {
            out.print ("<A HREF=\"" + servletUri +
                       "?view_media=image&id=" + id + "\">" +
                       "View non-Image Media (" +
                       img.getMimeType() + ")</A>" );
        }
        out.println( "</td></tr>" );
        out.println( "</table></p>" );

        //
        // Write the common HTML trailer
        //
        printPageTrailer( true );

        //
        // Close the result-set and the statement.
        //
        rset.close();
        stmt.close();
    }

    /**
     * Retrieve a thumbnail or full-size image from the database and
     * deliver it to the browser.
     */
    void viewMedia( String media )
        throws ServletException, IOException, SQLException
    {
        //
        // Prepare and execute a SQL statement to fetch the thumbnail or
        // full-size image. Although the SELECT statement is built on-the-fly,
        // depending on whether the thumbnail or full-size image is required,
        // this results in only 2 SQL statements being used (one for each
        // column), as the ID is specified as a parameter marker. Therefore,
        // the likelyhood is greater that the statements will be in the
        // server's cache.
        //
        PreparedStatement stmt =
            conn.prepareStatement( "select " + media +
                                   " from photos where id = ?" );
        stmt.setString( 1, request.getParameter( "id" ) );
        OracleResultSet rset = (OracleResultSet)stmt.executeQuery();

        //
        // Fetch the row from the result set.
        //
        if ( rset.next() )
        {
            //
            // Get the OrdImage object from the result set.
            //
            OrdImage img =
                 (OrdImage)rset.getORAData(1, OrdImage.getORADataFactory());

            //
            // Create an OrdHttpResponseHandler object, then use it to retrieve
            // the image from the database and deliver it to the browser.
            //
            OrdHttpResponseHandler handler =
                new OrdHttpResponseHandler( request, response );
            handler.sendImage( img );
        }
        else
        {
            //
            // Row not found, return a suitable error.
            //
            response.setStatus( response.SC_NOT_FOUND );
        }

        //
        // Close the result-set and the statement.
        //
        rset.close();
        stmt.close();
    }

    /**
     * The HTML upload form as a static String.
     */
    private static final String UPLOAD_FORM_FIELDS =
        "<table border=\"0\" width=\"100%\" cellspacing=\"0\"" +
        " summary=\"photo upload form\">\r\n" +
        " <tr bgcolor=\"#f7f7e7\">\r\n" +
        "  <td scope=\"col\" colspan=\"2\">&nbsp;</td>\r\n" +
        " </tr>\r\n" +
        " <tr bgcolor=\"#f7f7e7\">\r\n" +
        "  <td scope=\"col\" valign=\"top\"><label for=\"description_id\">" +
        "<b>Description:</b>\r\n" +
        "      <br><font size=\"-1\">(Optional)</font></label></td>\r\n" +
        "  <td scope=\"col\" ><input id=\"description_id\" type=\"text\" " +
        "   name=\"description\" length=\"40\">\r\n" +
        "      <br><font size=\"-1\">(e.g., My vacation)</font></td>\r\n" +
        " </tr>\r\n" +
        " <tr bgcolor=\"#f7f7e7\">\r\n" +
        "  <td scope=\"col\" valign=\"top\"><label for=\"location_id\">" +
        "<b>Location:</b>\r\n" +
        "      <br><font size=\"-1\">(Optional)</font></label></td>\r\n" +
        "  <td scope=\"col\" ><input id=\"location_id\" type=\"text\" " +
        " name=\"location\" length=\"40\">\r\n" +
        "      <br><font size=\"-1\">(e.g., Somewhere sunny)</font></td>\r\n" +
        " </tr>\r\n" +
        " <tr bgcolor=\"#f7f7e7\">\r\n" +
        "  <td scope=\"col\" valign=\"top\"><label for=\"filename_id\"><b>" +
        " File name:</b></label></td>\r\n" +
        "  <td scope=\"col\" ><input id=\"filename_id\" type=\"file\" " +
        " name=\"photo\">\r\n" +
        "      <br><font size=\"-1\">(e.g., island.jpg)</font></td>\r\n" +
        " </tr>\r\n" +
        " <tr bgcolor=\"#f7f7e7\">\r\n" +
        "  <td colspan=\"2\">&nbsp;</td>\r\n" +
        " </tr>\r\n" +
        " <tr bgcolor=\"#f7f7e7\">\r\n" +
        "  <td colspan=\"2\"><input type=\"submit\" value=\"Upload photo\" >" +
        " </td>\r\n" +
        " </tr>\r\n" +
        " <tr bgcolor=\"#f7f7e7\">\r\n" +
        "  <td colspan=\"2\">&nbsp;</td></tr>\r\n" +
        "</table>\r\n";

    /**
     * Display an HTML form that allows a new entry to be added to the album.
     */
    void viewUploadForm()
        throws ServletException, IOException
    {
        //
        // Output the page header, the form and the page trailer.
        //
        viewUploadForm( null );
    }

    /**
     * Display an HTML form that allows a new entry to be added to the album.
     */
    void viewUploadForm( String errorMessage )
        throws ServletException, IOException
    {
        //
        // Output the page header, the form and the page trailer.
        //
        printPageHeader();
        if ( errorMessage != null )
        {
            printMessage( "Error message", errorMessage );
        }

        out.println( "<p>");
        out.println( "<FONT SIZE=3 COLOR=\"#336699\"><B>Upload a new photo</B>");
        out.println( "</FONT><HR SIZE=1>" );
        out.println( "</p> ");
        out.println( "<p><form action=\"" + servletUri + "?upload=yes\"" );
        out.println( "      method=\"post\" enctype=\"multipart/form-data\">" );
        out.println( UPLOAD_FORM_FIELDS );
        out.println( "</form></p>" );
        printPageTrailer( true );
    }


    /**
     * This method is called by the doPost method to insert a new photo
     * into the album, creating a thumbnail image at the same time.
     */
    void insertNewPhoto()
        throws ServletException, IOException, SQLException
    {
        //
        // Create an OrdHttpUploadFormData object.
        //
        OrdHttpUploadFormData formData = new OrdHttpUploadFormData( request );

        //
        // Display the upload form if not an upload request.
        //
        if ( !formData.isUploadRequest() )
        {
            viewUploadForm();
            return;
        }

        //
        // Parse the multipart/form-data message.
        //
        formData.parseFormData();

        //
        // Get the description, location and photo.
        //
        String description = formData.getParameter( "description" );
        String location = formData.getParameter( "location" );
        OrdHttpUploadFile photo = formData.getFileParameter( "photo" );

        //
        // Make sure a valid image file was provided.
        //
        if ( photo == null ||
             photo.getOriginalFileName() == null ||
             photo.getOriginalFileName().length() == 0
           )
        {
            viewUploadForm( "Please supply a file name." );
            return;
        }

        if ( photo.getContentLength() == 0 )
        {
            viewUploadForm( "Please supply a valid image file." );
            return;
        }

        //
        // Use the file name if there is no description.
        //
        if ( description == null || description.length() == 0 )
        {
            description = "Image from file: " + photo.getSimpleFileName() + ".";
            if(description.length() > 40)
            {
                description = description.substring(0, 40);            
            }
        }

        //
        // We're going to be updating the database and writing to LOBs, so
        // make sure auto-commit is disabled.
        //
        conn.setAutoCommit( false );

        //
        // Get a value for the ID column of the new row
        //
        OraclePreparedStatement stmt =
            (OraclePreparedStatement)conn.prepareStatement(
                "select photos_sequence.nextval from dual" );
        OracleResultSet rset = (OracleResultSet)stmt.executeQuery();
        if ( !rset.next() )
        {
            throw new ServletException( "new ID not found" );
        }
        String id = rset.getString( 1 );
        rset.close();
        stmt.close();

        //
        // Prepare and execute a SQL statement to insert the new row.
        //
        stmt = (OraclePreparedStatement)conn.prepareStatement(
                "insert into photos (id,description,location,image,thumb) " +
                " values (?,?,?," + EMPTY_IMAGE + "," + EMPTY_IMAGE + ")" );
        stmt.setString( 1, id );
        stmt.setString( 2, description );
        stmt.setString( 3, location );
        stmt.executeUpdate();
        stmt.close();

        //
        // Prepare and execute a SQL statement to fetch the initialized
        // full-size and thumbnail image objects from the table.
        //
        stmt = (OraclePreparedStatement)conn.prepareStatement(
                    "select image,thumb from photos where id = ? for update" );
        stmt.setString( 1, id );
        rset = (OracleResultSet)stmt.executeQuery();
        if ( !rset.next() )
        {
            throw new ServletException( "new row not found in table" );
        }
        //
        OrdImage image =
            (OrdImage)rset.getORAData( 1, OrdImage.getORADataFactory());
        OrdImage thumb =
            (OrdImage)rset.getORAData( 2, OrdImage.getORADataFactory());

        rset.close();
        stmt.close();

        //
        // Load the photo into the database and set the properties.
        //
        photo.loadImage( image );

        //
        // Some image formats are supported by Oracle Multimedia but may not be
        // able to be displayed in-line by a browser. The BMP format is one 
        // example. Convert the image to a GIF or JPEG based on number of colors 
        // in the image.
        //
        if ( image.getContentFormat() != null &&
             image.getMimeType().indexOf( "bmp" ) > 0 )
        {
            try
            {
                image.process( "fileFormat=" +
                               getPreferredFormat( image.getContentFormat() ) );
            }
            catch ( SQLException e )
            {
            }
        }

        //
        // Try to copy the full-size image and process it to create the
        // thumbnail. This may not be possible if the image format is
        // not recognized.
        //
        try
        {
            image.processCopy( "maxScale=50,50", thumb );
        }
        catch ( SQLException e )
        {
            thumb.deleteContent();
            thumb.setContentLength( 0 );
        }

        //
        // Prepare and execute a SQL statement to update the full-size and
        // thumbnail images in the database.
        //
        stmt = (OraclePreparedStatement)conn.prepareStatement(
                    "update photos set image = ?, thumb = ? where id = ?" );
        //
        stmt.setORAData( 1, image );
        stmt.setORAData( 2, thumb );

        stmt.setString( 3, id );
        stmt.execute();
        stmt.close();

        //
        // Commit the changes.
        //
        conn.commit();

        //
        // Direct the browser to display the main page.
        //
        printPageHeader(
            "<meta http-equiv=\"refresh\" content=\"2 ;url=" + servletUri + "?view_album=yes\">" );
        printMessage( "Photo successfully uploaded into photo album",
                      "Please click on the link below or" +
                      " wait for the browser to refresh the page." );
        printPageTrailer( true );
    }


    /**
     * Display an HTML form that requires users to input DB username, 
     * password and connection string.
     *
     * Notes:
     * For simplicity in demonstrating this feature, this example does 
     * not perform the password management techniques that a deployed 
     * system normally uses. In a production environment, follow the 
     * Oracle Database password management guidelines, and disable any 
     * sample accounts. See Oracle Database Security Guide for password 
     * management guidelines and other security recommendations. 
     *
     */
    void viewDBInfoForm( String username, String connString, String errorMessage )
        throws ServletException, IOException
    {
        //
        // Output the page header, the form and the page trailer.
        //
        printPageHeader();
        out.println("<BR>");
        if ( errorMessage != null )
        {
            out.println("<p align=\"center\">");
            printErrorMessage(errorMessage );
            out.println("</p>");
        }
        //
        //disable the browser cache
        //
        response.setHeader( "Pragma", "no-cache" );
        response.setHeader( "Cache-Control", "no-cache" );
        response.setDateHeader( "Expires", 0 );

        String usernameValue = (username==null)?"":username;
        String connStringValue = (connString==null)?"":connString;
        out.println( "<form name=\"connInfo\" action=\"" + servletUri + 
                     "?page=dbInfo\"" );
        out.println( " method=\"post\" >" );
        out.println(DBINFO_FORM_FIELDS_1);
        out.println( "value=\""+usernameValue+"\">");
        out.println(DBINFO_FORM_FIELDS_2);
        out.println( "value=\""+connStringValue+"\">");
        out.println(DBINFO_FORM_FIELDS_3);
        out.println( "</form></p>" );
    }

    private static final String DBINFO_FORM_FIELDS_1 = 
        "<table border=\"0\" cellspacing=\"15\" align=\"center\" "+
        " bgcolor=\"#f7f7e7\"><tr><th colspan=\"2\"><font color=\"#336699\">"+
        "Please provide the following information for the application"+ 
        "<br>to connect to a database, retrieve and store images.</font>"+
        "</th></tr><tr><td>Schema:</td><td><input type=\"text\" "+
        " name=\"username\"  maxlength=\"30\"";

    private static final String DBINFO_FORM_FIELDS_2 =
        "</td></tr><tr><td>Password:</td><td><input type=\"password\"" +
        "  name=\"password\"  size=\"18\" maxlength=\"30\"></td></tr>"+
        "  <td>JDBC Connect String:<br><font size=\"-2\"></td><td>"+
        "  <input type=\"text\" name=\"connString\"";

    private static final String DBINFO_FORM_FIELDS_3 =
        "<br><font size=\"-1\">(See README.txt file for explanation)"+
        "</font></td></tr><tr>"+
        " <td align=\"right\" colspan=\"2\"><input type=\"submit\" "+
        " value=\"Submit\"></td></tr></table>";


    /**
     * Private function to return the preferred file format to which
     * image formats not supported by a browser should be converted.
     */
    private String getPreferredFormat( String contentFormat )
    {
        //
        // Image content format strings have the following format:
        //   <#bits><format>
        //   MONOCHROME
        //
        int numDigits = 0;
        while ( Character.isDigit( contentFormat.charAt( numDigits ) ) )
        {
            numDigits++;
        }

        //
        // Images with more than 8 bits of color can be converted to the JPEG
        // format without significant discernible loss of quality.
        //
        if ( numDigits > 0 )
        {
            if ( Integer.parseInt( contentFormat.substring( 0, numDigits ) ) > 8 )
            {
                return "JFIF";
            }
        }

        //
        // Images with 8 bits of color or less are best converted to the GIF
        // format to retain the quality.
        //
        return "GIFF";
    }

    /**
     * Display an HTML header that is common to all HTML responses.
     */
    private void printPageHeader()
        throws ServletException, IOException
    {
        printPageHeader( null );
    }

    /**
     * Display an HTML header that is common to all HTML responses.
     */
    private void printPageHeader( String extraHeader )
        throws ServletException, IOException
    {
        //
        // Get the PrintWriter output stream.
        //
        out = response.getWriter();

        //
        // Start the response.
        //
        response.setContentType( "text/html" );
        out.println( "<html lang=\"EN\">" );
        out.println( "<head>" );
        out.println( "<title>Oracle Multimedia Java Servlet Photo Album Demo</title>" );
        if ( extraHeader != null )
        {
            out.println( extraHeader );
        }
        out.println( "</head>" );
        out.println( "<body>" );
        out.println( "<table border=\"0\" width=\"100%\">" );
        out.println( "<tr>" );
        out.println( " <td colspan=\"2\" bgcolor=\"#f7f7e7\" align=\"center\">" );
        out.println( "  <font size=\"+2\">" );
        out.println( "   Oracle Multimedia Java Servlet Photo Album Demo" );
        out.println( "  </font>" );
        out.println( " </td>" );
        out.println( "</tr>" );
        out.println( "</table>" );
    }

    /**
     * Display an HTML trailer that is common to all HTML responses.
     */
    private void printPageTrailer( boolean displayReturnUrl )
        throws ServletException, IOException
    {
        if ( displayReturnUrl )
        {
            printLink( "?view_album=yes", "Return to photo album" );
        }
        out.println( "</body></html>" );
    }

    /**
     * Print a message
     */
    private void printMessage( String heading, String message )
    {
        printHeading( heading );
        out.println( "<p>" + message + "</p>" );
    }

    /**
     * Print a heading on a page
     */
    private void printHeading( String heading )
    {
        out.println( "<p><font size=3 color=\"#336699\">" );
        out.println( "<b>" + heading + "</b></font><hr size=1></p>" );
    }

    /**
     * Output an anchor tag in a standard format.
     */
    private void printLink( String queryString, String text )
    {
        out.println( "<p><table width=\"100%\"><tr bgcolor=\"#f7f7e7\">" );
        out.println( " <td colspan=\"3\" align=\"center\">" );
        out.println( "  <a href=\"" + servletUri + queryString + "\">" +
                     text + "</a>" );
        out.println( " </td>" );
        out.println( "</tr></table></p>" );
    }

    /**
     * Output an error message on the page.
     */
    private void printErrorMessage(String message){
        out.println("<img src=\"../../images/error_qualifier.png\"");
        out.println("name=\"errorIcon\" alt=\"Error:\"> " + message );
    }


    /**
     * Escape some most commonly used special chars in HTML.
     */
    public static final String escapeHtmlString(String input)
    {
        StringBuffer sb = new StringBuffer();

        for (int i = 0; i < input.length(); i++) 
        {
            char ch = input.charAt(i);
            switch (ch) 
            {
                case '<': 
                    sb.append("&lt;"); 
                    break;
                case '>': 
                    sb.append("&gt;"); 
                    break;
                case '&': 
                    sb.append("&amp;"); 
                    break; 
                case '"': 
                    sb.append("&quot;"); 
                    break;
                case ' ': 
                    sb.append("&nbsp;");
                    break;         
         
                default:  
                    sb.append(ch);
            }
        }
        
        return sb.toString();
    }
}

